/*
 * Routines to support Link Aggregation for ethernet
 */

#include <netinet/in.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_topo_map.h"
#include "lf_fma_flags.h"

#include "fma.h"
#include "fma_myri_packet.h"
#include "fma_fabric.h"
#include "fma_lag.h"

/*
 * Local routines
 */
static struct fma_lag_id_def *fma_lag_lookup_lag_id(
    struct fma_lag_id_def **hash_table, unsigned char *lag_id);

/*
 * Assign mag_ids to all NICs that need them
 */
void
fma_lag_assign_mag_ids(
  struct lf_fabric *fp)
{
  struct fma_lag_id_def *lidp;
  struct fma_lag_id_def *next_lidp;
  struct fma_lag_id_def **lag_hash;
  struct lf_host *hp;
  struct lf_nic *nicp;
  int next_mag_id;
  int this_mag_id;
  int h;
  int i;
  int n;

  /* Allocate hash table for lag_ids */
  LF_CALLOC(lag_hash, struct fma_lag_id_def *, FMA_LAG_HASH_TABLE_SIZE);

  /*
   * Loop through all NICs, for each one with a lag_id, check to see if
   * it is one we have seen before.  If not, create a new entry for it,
   * otherwise just increment the reference count.
   * In the end, only LAG_IDs with reference count >1 will receive
   * non-zero MAG_IDs.
   */
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    /* Check the LAG_ID of each NIC */
    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      /* only compute mag_id if host can do it */
      if (hp->fma_flags & FMA_FLAG_CAN_DO_LAG) {

	/* Look up LAG_ID descriptor, creating one if new */
	lidp = fma_lag_lookup_lag_id(lag_hash, FMA_NIC(nicp)->lag_id);

	/* Link this NIC onto it */
	FMA_NIC(nicp)->lag_next = lidp->lag_nic_list;
	lidp->lag_nic_list = nicp;

	/* up the reference count */
	++lidp->lag_ref_cnt;

      /* mag_id is zero if host cannot do LAG */
      } else {
	nicp->mag_id = 0;
      }
    }
  }

  /*
   * All lag_ids are now hashed - step through the hash table, assigning
   * mag_ids to all holders of lag_ids with reference counts > 1
   */
  next_mag_id = 1;
  for (i=0; i<FMA_LAG_HASH_TABLE_SIZE; ++i) {
    for (lidp = lag_hash[i];
	 lidp != NULL;
	 lidp = next_lidp) {

      /* If reference count <= 0, then mag_id is 0 */
      if (lidp->lag_ref_cnt <= 1) {
	this_mag_id = 0;
      } else {
	this_mag_id = next_mag_id;
	++next_mag_id;
      }

      /* Assign to all NICs in list */
      for (nicp = lidp->lag_nic_list;
	   nicp != NULL;
	   nicp = FMA_NIC(nicp)->lag_next) {
	nicp->mag_id = this_mag_id;
      }

      /* get pointer to next descriptor and free this one */
      next_lidp = lidp->lag_hash_next;
      LF_FREE(lidp);
    }
  }

#if FMA_TEST_LAG
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];

    for (n=0; n<hp->num_nics; ++n) {
      nicp = hp->nics[n];

      if (hp->fma_flags & FMA_FLAG_CAN_DO_LAG) {
	printf(LF_MAC_FORMAT ", LAG_ID=\"%s\", mag_id=%d\n",
	    LF_MAC_ARGS(nicp->mac_addr),
	    fma_lag_id_string(FMA_NIC(nicp)->lag_id),
	    nicp->mag_id);
      } else {
	printf(LF_MAC_FORMAT ", cannot LAG, mag_id=%d\n",
	    LF_MAC_ARGS(nicp->mac_addr), nicp->mag_id);
      }
    }
  }
#endif

  LF_FREE(lag_hash);
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Find a lag_id definition in our hash table, creating it if not already
 * existant.
 */
static struct fma_lag_id_def *
fma_lag_lookup_lag_id(
  struct fma_lag_id_def **hash_table,
  unsigned char *lag_id)
{
  struct fma_lag_id_def *lidp;
  int i;

  /* get hash index */
  i = fma_lag_hash(lag_id);

  /* loop through list of lag descriptors with same hash for this lag_id */
  for (lidp = hash_table[i];
       lidp != NULL;
       lidp = lidp->lag_hash_next) {

    /* If we find it, return */
    if (fma_lag_match(lidp->lag_id, lag_id)) {
      return lidp;
    }
  }

  /* not found, allocate a new one and put it in this hash bucket */
  LF_CALLOC(lidp, struct fma_lag_id_def, 1);
  lidp->lag_hash_next = hash_table[i];
  fma_lag_copy(lidp->lag_id, lag_id);
  hash_table[i] = lidp;

  return lidp;

 except:
  fma_perror_exit(1);
  return NULL;
}

/*
 * Generate a string representing a LAG_ID
 */
char *
fma_lag_id_string(
  unsigned char *lag_id)
{
  static lf_string_t s;

  sprintf(s, "[(%04x," LF_MAC_FORMAT ",%04x,%02x,%04x),"
  	     "(%04x," LF_MAC_FORMAT ",%04x,%02x,%04x)]",
	     ntohs(*(uint16_t *)(lag_id+0)),
	     LF_MAC_ARGS(lag_id+2),
	     ntohs(*(uint16_t *)(lag_id+8)),
	     *(lag_id+10),
	     ntohs(*(uint16_t *)(lag_id+11)),
	     ntohs(*(uint16_t *)(lag_id+13)),
	     LF_MAC_ARGS(lag_id+15),
	     ntohs(*(uint16_t *)(lag_id+21)),
	     *(lag_id+23),
	     ntohs(*(uint16_t *)(lag_id+24)));

  return s;
}
